<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Dashboard extends CI_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model(['Domain_model', 'Path_model', 'Migration_log_model']);
        $this->load->helper(['url', 'form', 'shortlink_helper']);
        $this->load->library('session');
        // Cek login
        if (!$this->session->userdata('logged_in'))
            redirect('auth/login');
    }



    public function index()
    {
        $data['total_domains'] = $this->Domain_model->count_all();

        // Use exact toko tables list from migration file to count paths
        $db = $this->load->database('', true); // Load database once and reuse
        $toko_tables = [
            'ompong188_links',
            'botak123_links',
            'sakti123_links',
            'pasar123_links',
            'cuan123_links',
            'jablay123_links',
            'candu123_links',
            'mampir123_links',
            'becekwin_links',
            'jenius_links',
            'gendut188_links'
        ];
        $total_paths = 0;
        foreach ($toko_tables as $table_name) {
            if ($db->table_exists($table_name)) {
                $count = $db->count_all($table_name);
                $total_paths += $count;
            }
        }
        // Also count paths from generic 'links' table if exists
        if ($db->table_exists('links')) {
            $total_paths += $db->count_all('links');
        }
        $data['total_paths'] = $total_paths;

        $data['migration_status'] = $this->Migration_log_model->get_status_summary();
        $data['migration_per_domain'] = $this->Migration_log_model->get_migration_per_domain();

        // Get latest shortlinks
        $data['latest_shortlinks'] = $this->get_latest_shortlinks();

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('dashboard', $data);
        $this->load->view('templates/footer');
    }

    private function get_latest_shortlinks($limit = 5)
    {
        // Get temporary shortlinks from session
        $temp_shortlinks = $this->session->userdata('temp_shortlinks') ?? [];

        // Clean up expired shortlinks (older than 3 days)
        $current_time = time();
        $three_days_ago = $current_time - (1 * 24 * 60 * 60);
        $temp_shortlinks = array_filter($temp_shortlinks, function($link) use ($three_days_ago) {
            return $link['created_at'] > $three_days_ago;
        });

        // Update session with cleaned shortlinks
        $this->session->set_userdata('temp_shortlinks', $temp_shortlinks);

        // Sort by created_at descending and limit
        usort($temp_shortlinks, function($a, $b) {
            return $b['created_at'] <=> $a['created_at'];
        });

        $limited_links = array_slice($temp_shortlinks, 0, $limit);

        // Convert to object format for compatibility with view
        $result = [];
        foreach ($limited_links as $link) {
            $obj = new stdClass();
            $obj->id = $link['id'];
            $obj->shortlink = $link['shortlink'];
            $obj->url = $link['url'];
            $obj->domain = $link['domain'];
            $obj->created_at = date('Y-m-d H:i:s', $link['created_at']);
            $obj->_table = 'session'; // Mark as session-stored
            $result[] = $obj;
        }

        return $result;
    }

    public function generate_shortlink()
    {
        if ($this->input->method() !== 'post') {
            show_error('Invalid request method.', 405);
        }

        $input_url = trim($this->input->post('url'));
        if (empty($input_url)) {
            $this->session->set_flashdata('error', 'URL is required.');
            redirect('dashboard');
        }

        // Call Cuttly API directly without domain existence check
        $shortlink = call_cuttly_api($input_url);
        $status = $shortlink ? 'success' : 'fail';

        if ($status === 'success') {
            // Parse URL to get domain and path for saving
            $parsed = parse_url($input_url);
            $domain_name = $parsed['host'] ?? null;
            $path = $parsed['path'] ?? null;
            if ($path === '/') {
                $path = null;
            }

            // Get existing temporary shortlinks from session
            $temp_shortlinks = $this->session->userdata('temp_shortlinks') ?? [];

            // Clean up expired shortlinks (older than 3 days)
            $current_time = time();
            $three_days_ago = $current_time - (1 * 24 * 60 * 60);
            $temp_shortlinks = array_filter($temp_shortlinks, function($link) use ($three_days_ago) {
                return $link['created_at'] > $three_days_ago;
            });

            // Generate unique ID for this shortlink
            $id = uniqid('temp_', true);

            // Add new shortlink to session
            $new_shortlink = [
                'id' => $id,
                'domain' => $domain_name ?? '',
                'url' => $input_url,
                'path' => $path,
                'shortlink' => $shortlink,
                'status' => $status,
                'created_at' => $current_time
            ];

            $temp_shortlinks[] = $new_shortlink;

            // Keep only latest 5 shortlinks
            if (count($temp_shortlinks) > 5) {
                // Sort by created_at descending and keep only first 5
                usort($temp_shortlinks, function($a, $b) {
                    return $b['created_at'] <=> $a['created_at'];
                });
                $temp_shortlinks = array_slice($temp_shortlinks, 0, 5);
            }

            // Save back to session
            $this->session->set_userdata('temp_shortlinks', $temp_shortlinks);

            $this->session->set_flashdata('success', 'Shortlink generated successfully.');
        } else {
            $this->session->set_flashdata('error', 'Failed to generate shortlink.');
        }

        redirect('dashboard');
    }

    public function remove_shortlink()
    {
        if ($this->input->method() !== 'post') {
            $this->output
                ->set_content_type('application/json')
                ->set_status_header(405)
                ->set_output(json_encode(['success' => false, 'message' => 'Invalid request method.']));
            return;
        }

        $id = $this->input->post('id');
        $table = $this->input->post('table');

        if (empty($id) || empty($table)) {
            $this->output
                ->set_content_type('application/json')
                ->set_status_header(400)
                ->set_output(json_encode(['success' => false, 'message' => 'Missing parameters.']));
            return;
        }

        // Handle session-stored temporary shortlinks
        if ($table === 'session') {
            $temp_shortlinks = $this->session->userdata('temp_shortlinks') ?? [];

            // Find and remove the shortlink with matching ID
            $found = false;
            foreach ($temp_shortlinks as $key => $link) {
                if ($link['id'] === $id) {
                    unset($temp_shortlinks[$key]);
                    $found = true;
                    break;
                }
            }

            if ($found) {
                // Re-index array and save back to session
                $temp_shortlinks = array_values($temp_shortlinks);
                $this->session->set_userdata('temp_shortlinks', $temp_shortlinks);

                $this->output
                    ->set_content_type('application/json')
                    ->set_output(json_encode(['success' => true]));
            } else {
                $this->output
                    ->set_content_type('application/json')
                    ->set_status_header(404)
                    ->set_output(json_encode(['success' => false, 'message' => 'Shortlink not found.']));
            }
            return;
        }

        // Handle database-stored shortlinks (existing functionality)
        // Validate table name to prevent SQL injection
        $allowed_tables = [
            'links',
            'ompong188_links',
            'botak123_links',
            'sakti123_links',
            'pasar123_links',
            'cuan123_links',
            'jablay123_links',
            'candu123_links',
            'mampir123_links',
            'becekwin_links',
            'jenius_links',
            'gendut188_links'
        ];

        if (!in_array($table, $allowed_tables)) {
            $this->output
                ->set_content_type('application/json')
                ->set_status_header(400)
                ->set_output(json_encode(['success' => false, 'message' => 'Invalid table name.']));
            return;
        }

        $db = $this->load->database('', true);

        // Delete the record by id from the specified table
        $deleted = $db->where('id', $id)->delete($table);

        if ($deleted) {
            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode(['success' => true]));
        } else {
            $this->output
                ->set_content_type('application/json')
                ->set_status_header(500)
                ->set_output(json_encode(['success' => false, 'message' => 'Failed to delete record.']));
        }
    }
}
