<?php
defined('BASEPATH') or exit('No direct script access allowed');

/**
 * @property CI_Input $input
 * @property CI_Session $session
 * @property CI_DB_query_builder $db
 * @property CI_Config $config
 * @property CI_Output $output
 * @property Domain_model $Domain_model
 * @property Path_model $Path_model
 */
class Pertoko extends CI_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->helper('url');
        $this->load->library('session');
        $this->load->model('Domain_model');
        $this->load->model('Path_model');

        // Cek login
        if (!$this->session->userdata('logged_in'))
            redirect('auth/login');
    }

    /* =========================
     * 🔹 Utility / Helper
     * ========================= */
    private function normalize_toko_slug($toko)
    {
        return (substr($toko, -4) !== '.com') ? $toko . '.com' : $toko;
    }

    private function validate_toko($toko)
    {
        return true; // bisa ditambahkan validasi jika perlu
    }

    private function get_table_name($toko)
    {
        $base_toko = strtolower($toko);
        if (substr($base_toko, -4) === '.com') {
            $base_toko = substr($base_toko, 0, -4);
        }
        $table_name = $base_toko . '_links';
        return $this->db->table_exists($table_name) ? $table_name : 'links';
    }

    private function find_domain($domain_name, $toko_id = null)
    {
        if ($toko_id) {
            return $this->Domain_model->get_domain_by_name_and_toko($domain_name, $toko_id);
        }
        $domains = $this->Domain_model->get_domains();
        foreach ($domains as $d) {
            if (strcasecmp($d->domain_name, $domain_name) === 0) {
                return $d;
            }
        }
        return null;
    }

    private function build_url($domain, $path)
    {
        if (!preg_match('#^https?://#i', $domain)) {
            $domain = 'https://' . $domain;
        }
        return rtrim($domain, '/') . '/' . ltrim($path, '/');
    }

    private function prepare_path_data($domain_obj, $path, $status = 'active', $keep_shortlink = null)
    {
        $full_url = $this->build_url($domain_obj->domain_name, $path);

        // Shortlink via Cuttly
        $shortlink = $keep_shortlink;
        if (!$shortlink) {
            $shortlink = $this->call_cuttly_api($full_url);
        }
        if (!$shortlink) {
            $shortlink = base_url($this->generate_shortlink($path));
        }

        return [
            'domain_id' => $domain_obj->id,
            'domain' => $domain_obj->domain_name,
            'path' => $path,
            'url' => $full_url,
            'shortlink' => $shortlink,
            'status' => $status,
            'created_at' => date('Y-m-d H:i:s')
        ];
    }

    private function call_cuttly_api($url, $api_key = null, $custom = null)
    {
        $api_key = $api_key ?? $this->config->item('cuttly_api_key');
        $api_url = $this->config->item('cuttly_api_url');

        $params = ['key' => $api_key, 'short' => $url];
        if ($custom) {
            $params['name'] = $custom;
        }
        $request_url = $api_url . '?' . http_build_query($params);

        $response = @file_get_contents($request_url);
        if ($response === FALSE)
            return null;

        $data = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE)
            return null;

        return $data['url']['shortLink'] ?? null;
    }

    private function generate_shortlink($path)
    {
        return 'shortlink/' . md5($path);
    }

    /* =========================
     * 🔹 Controller Actions
     * ========================= */
    public function index()
    {
        $toko_tables = [
            'ompong188',
            'botak123',
            'sakti123',
            'pasar123',
            'cuan123',
            'jablay123',
            'candu123',
            'mampir123',
            'becekwin',
            'jenius196',
            'gendut188'
        ];

        $data['toko_list'] = [];
        foreach ($toko_tables as $toko) {
            $table_name = $toko . '_links';
            if ($this->db->table_exists($table_name)) {
                $data['toko_list'][] = (object) ['domain' => $toko];
            }
        }

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('pertoko/index', $data);
        $this->load->view('templates/footer');
    }

    public function view($toko_slug)
    {
        // pastikan toko ada
        $toko = $this->db->get_where('toko', ['slug' => $toko_slug])->row();
        if (!$toko) {
            show_error('Toko tidak ditemukan: ' . $toko_slug, 404);
        }

        $search = $this->input->get('search');

        // ambil semua domain + link milik toko ini
        $this->db->select('l.*, d.domain_name')
            ->from('links l')
            ->join('domains d', 'l.domain_id = d.id')
            ->join('toko t', 'd.toko_id = t.id')
            ->where('t.id', $toko->id);

        if ($search) {
            $this->db->group_start();
            $this->db->like('d.domain_name', $search);
            $this->db->or_like('l.path', $search);
            $this->db->or_like('l.shortlink', $search);
            $this->db->group_end();
        }

        $paths = $this->db->get()->result();

        // siapkan data untuk view
        $data['domains'] = [];
        foreach ($paths as $p) {
            if (!isset($data['domains'][$p->domain_name])) {
                $data['domains'][$p->domain_name] = [
                    'name' => $p->domain_name,
                    'paths' => []
                ];
            }
            $data['domains'][$p->domain_name]['paths'][] = $p;
        }

        $data['all_domains'] = $this->Domain_model->get_domains_by_toko($toko->id);
        $data['toko'] = $toko->nama_toko;
        $data['toko_slug'] = $toko->slug;
        $data['search'] = $search;

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('pertoko/view', $data);
        $this->load->view('templates/footer');
    }

    public function edit($toko, $id)
    {
        $table_name = $this->get_table_name($toko);
        $data['link'] = $this->db->get_where($table_name, ['id' => $id])->row();
        if (!$data['link'])
            show_error('Data tidak ditemukan.', 404);

        $data['toko'] = ucfirst($toko);
        $data['toko_slug'] = $toko;

        if ($this->input->post())
            return $this->update($toko, $id);

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('pertoko/edit', $data);
        $this->load->view('templates/footer');
    }

    public function update($toko, $id)
    {
        $post = $this->input->post();
        $existing = $this->Path_model->get_by_id($id);
        if (!$existing)
            show_error('Data tidak ditemukan.', 404);

        $domain_obj = $this->find_domain($post['domain']);
        if (!$domain_obj)
            show_error('Domain not found.', 404);

        $update_data = $this->prepare_path_data($domain_obj, $post['path'], $post['status'], $existing->shortlink);
        $table_name = $this->get_table_name($toko);

        $this->db->where('id', $id)->update($table_name, $update_data);
        $this->session->set_flashdata('success', 'Data domain dan path pada ' . $toko . ' berhasil diperbarui.');
        redirect('pertoko/view/' . $toko);
    }

    public function delete($toko, $id)
    {
        $table_name = $this->get_table_name($toko);
        $this->db->where('id', $id)->delete($table_name);

        $this->session->set_flashdata('success', 'Domain dan path pada ' . $toko . ' berhasil dihapus.');
        redirect('pertoko/view/' . $toko);
    }

    public function add_domain($toko)
    {
        if ($this->input->post()) {
            $domain_name = strtolower(trim($this->input->post('domain_name')));
            if (!$domain_name) {
                show_error('Domain name is required.', 400);
            }

            // 🔹 pastikan toko ada
            $toko_row = $this->db->get_where('toko', ['slug' => $toko])->row();
            if (!$toko_row) {
                show_error('Toko tidak ditemukan: ' . $toko, 404);
            }
            $toko_id = $toko_row->id;

            // 🔹 simpan domain dengan relasi ke toko
            $this->db->insert('domains', [
                'domain_name' => $domain_name,
                'toko_id' => $toko_id,
                'status' => 'active',
                'created_at' => date('Y-m-d H:i:s')
            ]);

            $this->session->set_flashdata('success', 'Domain berhasil ditambahkan ke toko ' . ucfirst($toko_row->nama_toko));
            redirect('pertoko/view/' . $toko);
        }

        show_error('Invalid request method.', 405);
    }

    public function add_path($toko = null)
    {
        if ($this->input->method() === 'post') {
            $domain_name = $this->input->post('domain');
            $path = $this->input->post('path');
            $shortlink = $this->input->post('shortlink');

            // 🔹 pastikan toko ada
            $toko_row = $this->db->get_where('toko', ['slug' => $toko])->row();
            if (!$toko_row) {
                show_error('Toko tidak ditemukan: ' . $toko, 404);
            }
            $toko_id = $toko_row->id;

            $domain_obj = $this->find_domain($domain_name, $toko_id);
            if (!$domain_obj)
                show_error('Domain not found or does not belong to this store.', 404);

            $data = $this->prepare_path_data($domain_obj, $path, 'active', $shortlink);
            $table_name = $this->get_table_name($toko);
            $this->db->insert($table_name, $data);

            $this->session->set_flashdata('success', 'Domain dan path berhasil ditambahkan pada ' . $domain_obj->domain_name);
            redirect('pertoko/view/' . $toko);
        }

        // GET -> tampilkan form
        // 🔹 pastikan toko ada
        $toko_row = $this->db->get_where('toko', ['slug' => $toko])->row();
        if (!$toko_row) {
            show_error('Toko tidak ditemukan: ' . $toko, 404);
        }
        $toko_id = $toko_row->id;

        $domains = $this->Domain_model->get_domains_by_toko($toko_id);
        foreach ($domains as $domain) {
            $domain->domain = $domain->domain_name;
            $domain->name = $domain->domain_name;
        }
        $data['domains'] = $domains;
        $data['toko_slug'] = $toko;

        $this->load->view('templates/header');
        $this->load->view('templates/sidebar');
        $this->load->view('pertoko/add_path', $data);
        $this->load->view('templates/footer');
    }

    public function add_path_ajax($toko = null)
    {
        if ($this->input->method() !== 'post') {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode(['success' => false, 'message' => 'Invalid request method']));
        }

        $domain_name = $this->input->post('domain');
        $path = $this->input->post('path');
        $shortlink = $this->input->post('shortlink');
        if (empty($domain_name)) {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode(['success' => false, 'message' => 'Domain is required']));
        }

        // 🔹 pastikan toko ada
        $toko_row = $this->db->get_where('toko', ['slug' => $toko])->row();
        if (!$toko_row) {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode(['success' => false, 'message' => 'Toko tidak ditemukan']));
        }
        $toko_id = $toko_row->id;

        $domain_obj = $this->find_domain($domain_name, $toko_id);
        if (!$domain_obj) {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode(['success' => false, 'message' => 'Domain not found or does not belong to this store']));
        }

        $data = $this->prepare_path_data($domain_obj, $path, 'active', $shortlink);
        $table_name = $this->get_table_name($toko);
        $success = $this->db->insert($table_name, $data);

        $message = $success ? 'Path added successfully' : 'Failed to add path';
        $this->session->set_flashdata('success', 'Domain dan path berhasil ditambahkan pada ' . $domain_obj->domain_name);

        return $this->output->set_content_type('application/json')
            ->set_output(json_encode(['success' => $success, 'message' => $message]));
    }

    public function get_paths_by_domain($toko, $domain_name)
    {
        $table_name = $this->get_table_name($toko);

        if (!$this->db->table_exists($table_name)) {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode([]));
        }

        // ambil semua path dari tabel toko langsung (tanpa filter domain_id)
        $this->db->select('path')->from($table_name);
        $paths = $this->db->get()->result();

        return $this->output->set_content_type('application/json')
            ->set_output(json_encode($paths));
    }

    public function edit_path_all($toko)
    {
        if ($this->input->method() !== 'post')
            show_error('Invalid request method.', 405);

        $old_path = $this->input->post('old_path');
        $new_path = $this->input->post('new_path');
        if (!$old_path || !$new_path)
            show_error('Old path and new path are required.', 400);

        $table_name = $this->get_table_name($toko);
        $records = $this->db->where('path', $old_path)->get($table_name)->result();

        foreach ($records as $record) {
            $new_url = $this->build_url($record->domain, $new_path);
            $this->db->where('id', $record->id)->update($table_name, [
                'path' => $new_path,
                'url' => $new_url
            ]);
        }

        $this->session->set_flashdata('success', 'Path berhasil diubah dari "' . $old_path . '" ke "' . $new_path . '" pada ' . $toko);
        redirect('pertoko/view/' . $toko);
    }

    public function update_path_ajax($toko = null)
    {
        if ($this->input->method() !== 'post') {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode(['success' => false, 'message' => 'Invalid request method']));
        }

        $id = $this->input->post('id');
        $domain = $this->input->post('domain');
        $path = $this->input->post('path');
        $status = $this->input->post('status');
        if (empty($id) || empty($domain) || empty($path) || empty($status)) {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode(['success' => false, 'message' => 'All fields are required']));
        }

        $table_name = $this->get_table_name($toko);
        $existing = $this->db->get_where($table_name, ['id' => $id])->row();
        if (!$existing) {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode(['success' => false, 'message' => 'Data not found']));
        }

        $url = $this->build_url($domain, $path);
        $update_data = [
            'domain' => $domain,
            'path' => $path,
            'url' => $url,
            'shortlink' => $existing->shortlink,
            'status' => $status
        ];

        $success = $this->db->where('id', $id)->update($table_name, $update_data);
        $message = $success ? 'Path updated successfully' : 'Failed to update path';
        $this->session->set_flashdata('success', 'Data domain dan path pada ' . $toko . ' berhasil diperbarui.');

        return $this->output->set_content_type('application/json')
            ->set_output(json_encode(['success' => $success, 'message' => $message]));
    }

    public function generate_shortlink_ajax()
    {
        if ($this->input->method() !== 'post') {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode(['success' => false, 'message' => 'Invalid request method']));
        }

        $api_key = $this->input->post('api_key');
        $custom = $this->input->post('custom');
        $domain = $this->input->post('domain');
        $path = $this->input->post('path');

        if (empty($api_key) || empty($domain)) {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode(['success' => false, 'message' => 'API key and domain are required']));
        }

        $domain_obj = $this->find_domain($domain);
        if (!$domain_obj) {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode(['success' => false, 'message' => 'Domain not found']));
        }

        $full_url = $this->build_url($domain_obj->domain_name, $path ?? '');

        $shortlink = $this->call_cuttly_api($full_url, $api_key, $custom);

        if (!$shortlink) {
            return $this->output->set_content_type('application/json')
                ->set_output(json_encode(['success' => false, 'message' => 'Failed to generate shortlink']));
        }

        return $this->output->set_content_type('application/json')
            ->set_output(json_encode(['success' => true, 'shortlink' => $shortlink]));
    }
}
