// Auth JavaScript for modern features
document.addEventListener('DOMContentLoaded', function() {
    // Enhanced password toggle functionality
    const passwordToggles = document.querySelectorAll('.password-toggle');
    
    passwordToggles.forEach(toggle => {
        toggle.addEventListener('click', function() {
            const input = this.previousElementSibling;
            const type = input.getAttribute('type') === 'password' ? 'text' : 'password';
            input.setAttribute('type', type);
            
            // Toggle eye icon with smooth animation
            const eyeIcon = this.querySelector('svg');
            if (eyeIcon) {
                if (type === 'password') {
                    eyeIcon.innerHTML = '<path d="M16 8s-3-5.5-8-5.5S0 8 0 8s3 5.5 8 5.5S16 8 16 8zM1.173 8a13.133 13.133 0 0 1 1.66-2.043C4.12 4.668 5.88 3.5 8 3.5c2.12 0 3.879 1.168 5.168 2.457A13.133 13.133 0 0 1 14.828 8c-.058.087-.122.183-.195.288-.335.48-.83 1.12-1.465 1.755C11.879 11.332 10.119 12.5 8 12.5c-2.12 0-3.879-1.168-5.168-2.457A13.134 13.134 0 0 1 1.172 8z"/><path d="M8 5.5a2.5 2.5 0 1 0 0 5 2.5 2.5 0 0 0 0-5zM4.5 8a3.5 3.5 0 1 1 7 0 3.5 3.5 0 0 1-7 0z"/>';
                } else {
                    eyeIcon.innerHTML = '<path d="M13.359 11.238C15.06 9.72 16 8 16 8s-3-5.5-8-5.5a7.028 7.028 0 0 0-2.79.588l.77.771A5.944 5.944 0 0 1 8 3.5c2.12 0 3.879 1.168 5.168 2.457A13.134 13.134 0 0 1 14.828 8c-.058.087-.122.183-.195.288-.335.48-.83 1.12-1.465 1.755-.165.165-.337.328-.517.486l.708.709z"/><path d="M11.297 9.176a3.5 3.5 0 0 0-4.474-4.474l.823.823a2.5 2.5 0 0 1 2.829 2.829l.822.822zm-2.943 1.299.822.822a3.5 3.5 0 0 1-4.474-4.474l.823.823a2.5 2.5 0 0 0 2.829 2.829z"/><path d="M3.35 5.47c-.18.16-.353.322-.518.487A13.134 13.134 0 0 0 1.172 8l.195.288c.335.48.83 1.12 1.465 1.755C4.121 11.332 5.881 12.5 8 12.5c.716 0 1.39-.133 2.02-.36l.77.772A7.029 7.029 0 0 1 8 13.5C3 13.5 0 8 0 8s.939-1.721 2.641-3.238l.708.709zm10.296 8.884-12-12 .708-.708 12 12-.708.708z"/>';
                }
            }
        });
    });

    // Enhanced form validation and loading states
    const forms = document.querySelectorAll('form');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const submitBtn = this.querySelector('button[type="submit"]');
            
            // Enhanced validation with detailed feedback
            const inputs = this.querySelectorAll('input[required]');
            let isValid = true;
            
            inputs.forEach(input => {
                const value = input.value.trim();
                
                if (!value) {
                    isValid = false;
                    showInputError(input, 'Field ini wajib diisi');
                } else if (input.type === 'email' && !isValidEmail(value)) {
                    isValid = false;
                    showInputError(input, 'Format email tidak valid');
                } else if (input.name === 'password' && value.length < 8) {
                    isValid = false;
                    showInputError(input, 'Password minimal 8 karakter');
                }
            });
            
            if (!isValid) {
                e.preventDefault();
                return;
            }
            
            // Add loading state with animation
            submitBtn.classList.add('loading');
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<span class="loading-spinner"></span> Memproses...';
            
            // Remove loading state after form submission (this is handled by the form submission)
        });
    });

    // Enhanced input validation with real-time feedback
    const inputs = document.querySelectorAll('input');
    
    inputs.forEach(input => {
        // Real-time validation on input
        input.addEventListener('input', function() {
            clearInputError(this);
            
            if (this.name === 'password') {
                updatePasswordStrength(this.value);
            }
        });
        
        // Validation on blur
        input.addEventListener('blur', function() {
            const value = this.value.trim();
            
            if (this.hasAttribute('required') && !value) {
                showInputError(this, 'Field ini wajib diisi');
            } else if (this.type === 'email' && value && !isValidEmail(value)) {
                showInputError(this, 'Format email tidak valid');
            } else if (this.name === 'password' && value && value.length < 8) {
                showInputError(this, 'Password minimal 8 karakter');
            }
        });
        
        // Clear error on focus
        input.addEventListener('focus', function() {
            clearInputError(this);
        });
    });

    // Email validation function
    function isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }

    // Password strength indicator
    function updatePasswordStrength(password) {
        const strengthIndicator = document.getElementById('password-strength');
        if (!strengthIndicator) return;
        
        const strength = checkPasswordStrength(password);
        strengthIndicator.textContent = strength.text;
        strengthIndicator.className = `password-strength ${strength.class}`;
    }

    function checkPasswordStrength(password) {
        if (!password) return { text: '', class: '' };
        
        let strength = 0;
        const suggestions = [];
        
        // Length check
        if (password.length >= 8) strength++;
        else suggestions.push('minimal 8 karakter');
        
        // Uppercase check
        if (/[A-Z]/.test(password)) strength++;
        else suggestions.push('huruf besar');
        
        // Number check
        if (/[0-9]/.test(password)) strength++;
        else suggestions.push('angka');
        
        // Special character check
        if (/[^A-Za-z0-9]/.test(password)) strength++;
        else suggestions.push('karakter khusus');
        
        let text = '';
        switch(strength) {
            case 0:
            case 1:
                text = 'Lemah' + (suggestions.length ? ` - tambahkan ${suggestions.join(', ')}` : '');
                return { text, class: 'weak' };
            case 2:
                text = 'Sedang' + (suggestions.length ? ` - ${suggestions.join(', ')}` : '');
                return { text, class: 'medium' };
            case 3:
                text = 'Kuat' + (suggestions.length ? ` - ${suggestions.join(', ')}` : '');
                return { text, class: 'strong' };
            case 4:
                text = 'Sangat Kuat ✓';
                return { text, class: 'very-strong' };
            default:
                return { text: '', class: '' };
        }
    }

    // Error handling functions
    function showInputError(input, message) {
        clearInputError(input);
        
        input.style.borderColor = '#e53e3e';
        input.style.boxShadow = '0 0 0 4px rgba(229, 62, 62, 0.1)';
        
        const errorDiv = document.createElement('div');
        errorDiv.className = 'input-error';
        errorDiv.style.color = '#e53e3e';
        errorDiv.style.fontSize = '0.8rem';
        errorDiv.style.marginTop = '0.5rem';
        errorDiv.style.fontWeight = '500';
        errorDiv.textContent = message;
        
        input.parentNode.appendChild(errorDiv);
    }

    function clearInputError(input) {
        input.style.borderColor = '#e2e8f0';
        input.style.boxShadow = '0 2px 8px rgba(0, 0, 0, 0.05)';
        
        const existingError = input.parentNode.querySelector('.input-error');
        if (existingError) {
            existingError.remove();
        }
    }

    // Add smooth scrolling to form elements
    const formGroups = document.querySelectorAll('.form-group');
    formGroups.forEach((group, index) => {
        group.style.animationDelay = `${0.1 + (index * 0.1)}s`;
    });

    // Add focus styles for better accessibility
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Tab') {
            document.body.classList.add('keyboard-nav');
        }
    });

    document.addEventListener('mousedown', function() {
        document.body.classList.remove('keyboard-nav');
    });
});

// Add enhanced CSS styles
const enhancedStyles = document.createElement('style');
enhancedStyles.textContent = `
.input-error {
    color: #e53e3e !important;
    font-size: 0.8rem !important;
    margin-top: 0.5rem !important;
    font-weight: 500 !important;
    animation: slideDown 0.3s ease !important;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.loading-spinner {
    display: inline-block;
    width: 16px;
    height: 16px;
    border: 2px solid rgba(255, 255, 255, 0.3);
    border-radius: 50%;
    border-top-color: white;
    animation: spin 1s ease-in-out infinite;
    margin-right: 0.5rem;
}

/* Enhanced focus styles for keyboard navigation */
body.keyboard-nav .form-control:focus {
    outline: 2px solid #667eea;
    outline-offset: 2px;
}

body.keyboard-nav .btn-auth:focus {
    outline: 2px solid #667eea;
    outline-offset: 2px;
}

/* Smooth transitions for all interactive elements */
.form-control,
.btn-auth,
.password-toggle {
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

/* Enhanced hover effects */
.form-control:hover {
    border-color: #cbd5e0;
    transform: translateY(-1px);
}

.password-toggle:hover {
    transform: translateY(-50%) scale(1.1);
}

/* Success state for valid inputs */
.form-control.valid {
    border-color: #48bb78;
    box-shadow: 0 0 0 4px rgba(72, 187, 120, 0.1);
}

/* Error state for invalid inputs */
.form-control.error {
    border-color: #e53e3e;
    box-shadow: 0 0 0 4px rgba(229, 62, 62, 0.1);
}
`;
document.head.appendChild(enhancedStyles);
